/*
 * This is a simple example of a custom node that processes input text, intended to serve as a boilerplate for your own applications.
 * See more details by following https://docs.inworld.ai/docs/node/quickstart
 */

import 'dotenv/config';

import {
  CustomNode,
  GraphBuilder,
  GraphTypes,
  ProcessContext,
} from '@inworld/runtime/graph';
import { v4 } from 'uuid';

interface CustomTextOutput {
  processedText: string;
}

// Define a custom node that processes the input text
class CustomTextNode extends CustomNode {
  async process(
    context: ProcessContext,
    input: string,
  ): Promise<GraphTypes.Custom<CustomTextOutput>> {
    return {
      processedText: input.toUpperCase(),
    };
  }
}

// Create an instance of the custom node
const customTextNode = new CustomTextNode();

// Creating a graph builder instance and adding the node to it
const graphBuilder = new GraphBuilder('custom-text-node')
  .addNode(customTextNode)
  .setStartNode(customTextNode)
  .setEndNode(customTextNode);

// Creating an executor instance from the graph builder
const executor = graphBuilder.build();

main();

// Main function that executes the graph
async function main() {
  // Execute graph and waiting for output stream to be returned.
  const outputStream = await executor.start('Hello, world!', v4());

  // Unwrapping the output stream to get the first result (in our case it's the only result).
  const result = await outputStream.next();
  // If it is, we can access the data property of the result.
  process.stdout.write(
    `Graph execution result: ${JSON.stringify(result.data, null, 2)}\n`,
  );

  // Cleaning up the executor and destroying the graph builder.
  executor.cleanupAllExecutions();
  executor.destroy();
}
